/* $Id: srcparb.c,v 1.7 1998/07/20 22:40:44 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include <math.h>
#include "e1432.h"

#define	BLOCKSIZE	8192
#define	SIGLENGTH	8192
#define	SIGPERIOD	8192/4
#define	XFRLENGTH	4096
#define SRCBUFRSIZE	4096
#define MANUAL_ARM	1	/* 0 = autoarm/trig; 1 = manual arm/trig */

#define	SPAN		20000.0

#define	IN_MAX		32
#define	SRC_MAX		5

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Make these global  */
static E1432ID hw;
static SHORTSIZ16 group;
static SHORTSIZ16 chan_list[IN_MAX + SRC_MAX];
static int chan_count;

static SHORTSIZ16 in_list[IN_MAX];
static SHORTSIZ16 in_group;
static int in_count;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;


/******************* arb source routines **************************/

#define SIG_LENGTH_MAX 100000

static LONGSIZ32 src_sigbuff[SIG_LENGTH_MAX];
static LONGSIZ32 src_siglength;

static int src_sigperiod;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;

void
src_sig( LONGSIZ32 siglength, int sigperiod, long *sigbuff)
{

int i;

	  /* Synth a sine */
          for(i=0;i<siglength/2;i++)
	  {
             sigbuff[i] = (long)((float) 0x7fffffffL *
                          sin(2.0*M_PI*(float)i/((float)sigperiod)));
          }
          for(i=siglength/2;i<siglength;i++)
	  {
             sigbuff[i] = (long)((float) 0x7fffffffL *
                          sin(2.0*M_PI*(float)i/((float)sigperiod/(float)4)));
          }
}

void
xfr_src_sig(E1432ID lhw, SHORTSIZ16 chanID, long *sigbuff, int xfrlength, SHORTSIZ16 mode)
{
    LONGSIZ32 i, maxtries;
    short error;

	/* write xfrlength words to the substrate */

	i = 0;
	maxtries = 100000;

	for (i = 0; i < maxtries; i++)
	{
	    if (e1432_check_src_arbrdy(lhw, chanID, mode) == 1)
	    {
		error = e1432_write_srcbuffer_data(lhw, chanID,
						  sigbuff,
						  (SHORTSIZ16)  xfrlength,
						  mode);
		if(error)
		    (void) printf("Error %d in e1432_write_srcbuffer_data\n",
				  error);
		break;
	    }
	}
	if (i == maxtries)
	{
	    (void) printf("xfr_src_sig: timed out \n");
	    return;
	}
}

/***************** end arb source routines ************************/

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    (void) printf("read_data\n");
    /* Read some data */
    for (i = 0; i < in_count; i++)
    {
	CHECK(e1432_read_float64_data(hw, in_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
	 (void) printf("channel %d    data points read =  %d\n", i , count);
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }
    return 0;
}

int
main(void)
{
    int     i, error;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;
    int wait, nowait, manual_arm;
    int doitforever;
    wait = 1;
    nowait = 0;
    manual_arm = MANUAL_ARM;
    
    /* Initialize the library */
    CHECK(e1432_init_io_driver());

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(0);

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */
    error = e1432_get_hwconfig(1, &laddr, &hwconfig);
    if (error != 0)
    {
	(void) printf("downloading /opt/e1432/lib/sema.bin\n");
        error = e1432_install(1, &laddr, 0,
                              "/opt/e1432/lib/sema.bin");
        if (error)
	{
            (void) printf("e1432_install failed, error: %d\n", error);
	    return -1;
	}
	
	CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    }

    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create input channel group */
    if ( hwconfig.input_chans > 0)
    {
	in_count = hwconfig.input_chans;
	if (in_count > IN_MAX)
	    in_count = IN_MAX;
	for (i = 0; i < in_count; i++)
	    in_list[i] = E1432_INPUT_CHAN(i+1);
	in_group = e1432_create_channel_group(hw, in_count, in_list);
	
	if (in_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				in_group));
	    return -1;
	}
    }
    else
    {
	in_count =0;
    }

    /* Create source group */
    if ( hwconfig.source_chans > 0)
    {
	src_count = hwconfig.source_chans;
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
	for (i = 0; i < src_count; i++)
	    src_list[i] = E1432_SOURCE_CHAN(i+1);
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    src_group));
	    return -1;
	}
    }
    else
    {
	src_count =0;
    }

    /* Create global channel group */
    chan_count = 0;
    for (i = 0; i < in_count; i++)
    {
	chan_list[chan_count] = in_list[i];
	chan_count = chan_count+1;
    }
    for (i = 0; i < src_count; i++)
    {
	chan_list[chan_count] = src_list[i];
	chan_count = chan_count+1;
    }
    if (chan_count > 0)
    { 
	group = e1432_create_channel_group(hw, chan_count, chan_list);
	if (group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				group));
	    exit(1);
	}
    }

/* initialize all channels */
    CHECK(e1432_set_clock_freq(hw, group,51200));
    CHECK(e1432_set_span(hw, group, SPAN));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));

    (void) printf("manual_arm (1=manual, 0 = auto) %d\n",manual_arm);
    if (manual_arm)
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_MANUAL_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_MANUAL_ARM));
    }
    else
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_AUTO_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_AUTO_ARM));
    }

/* initialize inputs */
    CHECK(e1432_set_active(hw, in_group, E1432_CHANNEL_ON));

/* initialize source */
    if (src_count > 0)
    {

	arb_mode = E1432_SRCBUFFER_PERIODIC_AB;
/*
	arb_mode = E1432_SRCBUFFER_PERIODIC_A;
*/

	if ((arb_mode == E1432_SRCBUFFER_PERIODIC_AB)||(arb_mode == E1432_SRCBUFFER_CONTINUOUS))
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_AB;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;
	}
	else
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_A;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITA;
	}
	
	
	arb_srcbufsize = SRCBUFRSIZE;
	src_siglength = SIGLENGTH;
	src_sigperiod = SIGPERIOD;
	
	arb_xfrsize = XFRLENGTH;

	(void) printf("arb_xfrsize %d, arb_srcbufsize %d, src_siglength %d\n",
		      arb_xfrsize, arb_srcbufsize, src_siglength);
	(void) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
		      arb_xfrmode, arb_xfrmodewait);

	CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

    if (manual_arm)
    {
	CHECK(e1432_set_data_mode(hw, in_group, E1432_BLOCK_MODE));
	CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_BARB));
	CHECK(e1432_set_duty_cycle(hw, src_group, 1.0));
	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	{
	    CHECK(e1432_set_source_blocksize(hw, src_group, 2*SRCBUFRSIZE));
	}
	else
	{
	    CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
	}
    }
    else
    {
	CHECK(e1432_set_data_mode(hw, in_group, E1432_CONTINUOUS_MODE));
	CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_ARB));
	CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
    }

	CHECK(e1432_set_ramp_rate(hw, src_group, 0));
	CHECK(e1432_set_range(hw, src_group, 1.0));
/*	CHECK(e1432_set_span(hw, src_group, 625)); */
	CHECK(e1432_set_amp_scale(hw, src_group, 1.0));

	CHECK(e1432_auto_zero(hw, src_group));

	CHECK(e1432_set_srcbuffer_mode(hw, src_group, arb_mode));
	CHECK(e1432_set_srcbuffer_size(hw, src_group, arb_srcbufsize));
	CHECK(e1432_set_srcbuffer_init(hw, src_group,
				       E1432_SRCBUFFER_INIT_EMPTY));

/** create signal array ***/

	src_sig(src_siglength, src_sigperiod, &src_sigbuff[0]);

	/* down load signal */
	(void) printf("preload buffer A\n");
	xfr_src_sig(hw, src_list[0], &src_sigbuff[0],arb_xfrsize,arb_xfrmodewait);
	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB && (e1432_check_src_arbrdy(hw, src_list[0], arb_xfrmodewait)))
	{
	    /* load B as well as A */
	    (void) printf("preload buffer B\n");
	xfr_src_sig(hw, src_list[0], &src_sigbuff[arb_xfrsize -1],arb_xfrsize,arb_xfrmodewait);
	}
	(void) printf("done pre-loading buffer(s)\n");

    }
    else
    {
	(void) printf("Error : No source channels found\n");
	/* exit(1); */
    }

/* Start measurement */
    (void) printf("init_measure\n");
    CHECK(e1432_init_measure(hw, group));

    doitforever = 1;
    
    while(doitforever)
    {
	if (manual_arm)
	{
	    (void) printf("arm_measure \n");
	    CHECK(e1432_arm_measure(hw, group, wait));
	    (void) printf("trigger_measure \n");
	    CHECK(e1432_trigger_measure(hw, group, nowait));
	}

/* Wait for group block ready */
	error = 0;
	while (!error) /* Note ; */
	{
	    (void) printf("checking block_available \n");
	    error = e1432_block_available(hw, in_group);
	    if (error < 0)
	    {
		(void) printf("e1432_block_available() failed, error: %d\n", error);
		return -1;
	    }
	    if (error > 0)
	    {
		CHECK(read_data());
		break;
	    }
	}	
    }

    /*NOTREACHED*/
	return 0;
}

